/**
 * SVJ file format loader
 * Parses HabitatView .svj project files
 */
/** SVJ file metadata */
interface SvjMetadata {
    version: string;
    app_name: string;
    created_at: string;
    modified_at: string;
}
/** Project info */
interface SvjProjectInfo {
    name: string;
    selected_floor_index: number;
}
/** Sensor configuration */
interface SensorConfig {
    url?: string;
    entity_id?: string;
    unit?: string;
}
/** Room configuration */
interface RoomConfig {
    name: string;
    center_x?: number;
    center_y?: number;
    sensors?: Record<string, SensorConfig>;
}
/** Floor configuration */
interface FloorConfig {
    id: string;
    name: string;
    rooms?: Record<string, RoomConfig>;
}
/** Home Assistant configuration */
interface HomeAssistantConfig {
    url?: string;
    token?: string;
}
/** Full config.json structure */
interface SvjConfig {
    poll_interval?: number;
    home_assistant?: HomeAssistantConfig;
    floors?: FloorConfig[];
}
/** A floor with its SVG content */
interface SvjFloor {
    id: string;
    name: string;
    svg: string;
    config?: FloorConfig;
}
/** Parsed SVJ project */
interface SvjProject {
    name: string;
    metadata: SvjMetadata;
    config: SvjConfig;
    floors: SvjFloor[];
    selectedFloorIndex: number;
}
/**
 * Load and parse an SVJ file from an ArrayBuffer
 */
declare function parseSvj(buffer: ArrayBuffer): SvjProject;
/**
 * Load an SVJ file from a URL
 */
declare function loadSvjFromUrl(url: string): Promise<SvjProject>;
/**
 * Load an SVJ file from a File object (e.g., from file input)
 */
declare function loadSvjFromFile(file: File): Promise<SvjProject>;

/**
 * SVJ Viewer Component
 * Renders floor plan SVGs with pan/zoom support and sensor display
 */

/** Sensor value with metadata */
interface SensorValue {
    value: string | number;
    unit?: string;
    timestamp?: Date;
}
/** Viewer options */
interface ViewerOptions {
    /** Container element or CSS selector */
    container: HTMLElement | string;
    /** Background color */
    backgroundColor?: string;
    /** Enable pan/zoom interactions */
    interactive?: boolean;
    /** Fit content to viewport on load */
    fitOnLoad?: boolean;
    /** Minimum zoom level */
    minZoom?: number;
    /** Maximum zoom level */
    maxZoom?: number;
}
/** Event types */
type ViewerEventType = 'load' | 'floorChange' | 'click' | 'error' | 'sensorUpdate';
type ViewerEventHandler = (data: unknown) => void;
/**
 * SVJ Floor Plan Viewer
 */
declare class SvjViewer {
    private container;
    private wrapper;
    private svgContainer;
    private sensorOverlay;
    private options;
    private project;
    private currentFloorIndex;
    private viewport;
    private eventHandlers;
    private sensors;
    private pollingInterval;
    private isPolling;
    private isDragging;
    private lastMouseX;
    private lastMouseY;
    constructor(options: ViewerOptions);
    private setupInteractions;
    private zoomAt;
    private updateTransform;
    private injectStyles;
    private getSensorIcon;
    /**
     * Load an SVJ project from a URL
     */
    load(url: string): Promise<void>;
    /**
     * Load an SVJ project from a File object
     */
    loadFile(file: File): Promise<void>;
    /**
     * Load from an already-parsed project
     */
    loadProject(project: SvjProject): void;
    private renderCurrentFloor;
    private renderSensors;
    private createSensorElement;
    private getSensorTypeClass;
    private getDefaultUnit;
    /**
     * Fit the content to the viewport
     */
    fitToView(padding?: number): void;
    /**
     * Reset zoom and position
     */
    resetView(): void;
    /**
     * Set the current floor by index
     */
    setFloorIndex(index: number): void;
    /**
     * Set the current floor by ID
     */
    setFloor(id: string): void;
    /**
     * Get the current floor
     */
    getCurrentFloor(): SvjFloor | null;
    /**
     * Get all floors
     */
    getFloors(): SvjFloor[];
    /**
     * Get the project
     */
    getProject(): SvjProject | null;
    /**
     * Get current zoom level
     */
    getZoom(): number;
    /**
     * Set zoom level
     */
    setZoom(scale: number): void;
    /**
     * Add event listener
     */
    on(event: ViewerEventType, handler: ViewerEventHandler): void;
    /**
     * Remove event listener
     */
    off(event: ViewerEventType, handler: ViewerEventHandler): void;
    private emit;
    /**
     * Update a sensor value
     * @param roomId The room ID
     * @param sensorType The sensor type (e.g., 'temperature', 'humidity')
     * @param value The sensor value
     * @param unit Optional unit override
     */
    setSensorValue(roomId: string, sensorType: string, value: string | number, unit?: string): void;
    /**
     * Update multiple sensor values at once
     * @param values Map of "roomId:sensorType" to value
     */
    setSensorValues(values: Record<string, string | number>): void;
    /**
     * Get all sensor configurations for the current floor
     */
    getSensors(): Array<{
        roomId: string;
        sensorType: string;
        config: SensorConfig;
        value?: SensorValue;
    }>;
    /**
     * Get sensor value
     */
    getSensorValue(roomId: string, sensorType: string): SensorValue | undefined;
    /**
     * Start polling sensor values at the configured interval
     */
    startPolling(): void;
    /**
     * Stop polling sensor values
     */
    stopPolling(): void;
    /**
     * Fetch all sensor values from their configured URLs
     */
    fetchAllSensors(): Promise<void>;
    /**
     * Fetch a single sensor value
     */
    private fetchSensorValue;
    /**
     * Check if polling is active
     */
    isPollingActive(): boolean;
    /**
     * Destroy the viewer and clean up
     */
    destroy(): void;
}

/**
 * Simple tar archive parser
 * Handles reading tar files in the browser
 */
interface TarEntry {
    name: string;
    size: number;
    data: Uint8Array;
}
/**
 * Parse a tar archive from an ArrayBuffer
 */
declare function parseTar(buffer: ArrayBuffer): TarEntry[];
/**
 * Get a file from tar entries by name
 */
declare function getEntryByName(entries: TarEntry[], name: string): TarEntry | undefined;
/**
 * Get file content as string
 */
declare function entryAsString(entry: TarEntry): string;
/**
 * Get all entries matching a pattern
 */
declare function getEntriesMatching(entries: TarEntry[], pattern: RegExp): TarEntry[];

export { type FloorConfig, type HomeAssistantConfig, type RoomConfig, type SensorConfig, type SensorValue, type SvjConfig, type SvjFloor, type SvjMetadata, type SvjProject, type SvjProjectInfo, SvjViewer, type TarEntry, type ViewerEventHandler, type ViewerEventType, type ViewerOptions, entryAsString, getEntriesMatching, getEntryByName, loadSvjFromFile, loadSvjFromUrl, parseSvj, parseTar };
