"use strict";
var __defProp = Object.defineProperty;
var __getOwnPropDesc = Object.getOwnPropertyDescriptor;
var __getOwnPropNames = Object.getOwnPropertyNames;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __export = (target, all) => {
  for (var name in all)
    __defProp(target, name, { get: all[name], enumerable: true });
};
var __copyProps = (to, from, except, desc) => {
  if (from && typeof from === "object" || typeof from === "function") {
    for (let key of __getOwnPropNames(from))
      if (!__hasOwnProp.call(to, key) && key !== except)
        __defProp(to, key, { get: () => from[key], enumerable: !(desc = __getOwnPropDesc(from, key)) || desc.enumerable });
  }
  return to;
};
var __toCommonJS = (mod) => __copyProps(__defProp({}, "__esModule", { value: true }), mod);

// src/index.ts
var src_exports = {};
__export(src_exports, {
  SvjViewer: () => SvjViewer,
  entryAsString: () => entryAsString,
  getEntriesMatching: () => getEntriesMatching,
  getEntryByName: () => getEntryByName,
  loadSvjFromFile: () => loadSvjFromFile,
  loadSvjFromUrl: () => loadSvjFromUrl,
  parseSvj: () => parseSvj,
  parseTar: () => parseTar
});
module.exports = __toCommonJS(src_exports);

// src/tar.ts
function parseTar(buffer) {
  const entries = [];
  const view = new Uint8Array(buffer);
  let offset = 0;
  while (offset < view.length - 512) {
    const header = view.slice(offset, offset + 512);
    if (isZeroBlock(header)) {
      break;
    }
    const name = readString(header, 0, 100).replace(/\0/g, "");
    const sizeStr = readString(header, 124, 12).replace(/\0/g, "").trim();
    const size = parseInt(sizeStr, 8) || 0;
    offset += 512;
    if (name && size > 0) {
      const data = view.slice(offset, offset + size);
      entries.push({ name, size, data: new Uint8Array(data) });
    }
    offset += Math.ceil(size / 512) * 512;
  }
  return entries;
}
function readString(data, offset, length) {
  const bytes = data.slice(offset, offset + length);
  return new TextDecoder("utf-8").decode(bytes);
}
function isZeroBlock(block) {
  for (let i = 0; i < block.length; i++) {
    if (block[i] !== 0) return false;
  }
  return true;
}
function getEntryByName(entries, name) {
  return entries.find((e) => e.name === name);
}
function entryAsString(entry) {
  return new TextDecoder("utf-8").decode(entry.data);
}
function getEntriesMatching(entries, pattern) {
  return entries.filter((e) => pattern.test(e.name));
}

// src/svj.ts
function parseSvj(buffer) {
  const entries = parseTar(buffer);
  const metadataEntry = getEntryByName(entries, "metadata.json");
  const metadata = metadataEntry ? JSON.parse(entryAsString(metadataEntry)) : { version: "1.0", app_name: "HabitatView", created_at: "", modified_at: "" };
  const projectEntry = getEntryByName(entries, "project.json");
  const projectInfo = projectEntry ? JSON.parse(entryAsString(projectEntry)) : { name: "Untitled", selected_floor_index: 0 };
  const configEntry = getEntryByName(entries, "config.json");
  const config = configEntry ? JSON.parse(entryAsString(configEntry)) : {};
  const svgEntries = getEntriesMatching(entries, /^floors\/.*\.svg$/);
  const floors = svgEntries.map((entry) => {
    const match = entry.name.match(/^floors\/(.+)\.svg$/);
    const id = match ? match[1] : entry.name;
    const floorConfig = config.floors?.find((f) => f.id === id);
    return {
      id,
      name: floorConfig?.name ?? id,
      svg: entryAsString(entry),
      config: floorConfig
    };
  });
  return {
    name: projectInfo.name,
    metadata,
    config,
    floors,
    selectedFloorIndex: projectInfo.selected_floor_index
  };
}
async function loadSvjFromUrl(url) {
  const response = await fetch(url);
  if (!response.ok) {
    throw new Error(`Failed to load SVJ file: ${response.status} ${response.statusText}`);
  }
  const buffer = await response.arrayBuffer();
  return parseSvj(buffer);
}
async function loadSvjFromFile(file) {
  const buffer = await file.arrayBuffer();
  return parseSvj(buffer);
}

// src/viewer.ts
var SvjViewer = class {
  constructor(options) {
    this.project = null;
    this.currentFloorIndex = 0;
    this.viewport = { x: 0, y: 0, scale: 1 };
    this.eventHandlers = /* @__PURE__ */ new Map();
    this.sensors = /* @__PURE__ */ new Map();
    this.pollingInterval = null;
    this.isPolling = false;
    // Interaction state
    this.isDragging = false;
    this.lastMouseX = 0;
    this.lastMouseY = 0;
    if (typeof options.container === "string") {
      const el = document.querySelector(options.container);
      if (!el || !(el instanceof HTMLElement)) {
        throw new Error(`Container not found: ${options.container}`);
      }
      this.container = el;
    } else {
      this.container = options.container;
    }
    this.options = {
      container: this.container,
      backgroundColor: options.backgroundColor ?? "#ffffff",
      interactive: options.interactive ?? true,
      fitOnLoad: options.fitOnLoad ?? true,
      minZoom: options.minZoom ?? 0.1,
      maxZoom: options.maxZoom ?? 10
    };
    this.wrapper = document.createElement("div");
    this.wrapper.style.cssText = `
      position: absolute;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      overflow: hidden;
      background: ${this.options.backgroundColor};
    `;
    this.svgContainer = document.createElement("div");
    this.svgContainer.style.cssText = `
      position: absolute;
      transform-origin: 0 0;
      cursor: grab;
      left: 0;
      top: 0;
    `;
    this.sensorOverlay = document.createElement("div");
    this.sensorOverlay.style.cssText = `
      position: absolute;
      top: 0;
      left: 0;
      pointer-events: none;
    `;
    this.sensorOverlay.className = "svj-sensor-overlay";
    this.svgContainer.appendChild(this.sensorOverlay);
    this.wrapper.appendChild(this.svgContainer);
    this.container.appendChild(this.wrapper);
    this.injectStyles();
    if (this.options.interactive) {
      this.setupInteractions();
    }
  }
  setupInteractions() {
    this.wrapper.addEventListener("mousedown", (e) => {
      if (e.button === 0) {
        this.isDragging = true;
        this.lastMouseX = e.clientX;
        this.lastMouseY = e.clientY;
        this.svgContainer.style.cursor = "grabbing";
      }
    });
    window.addEventListener("mousemove", (e) => {
      if (this.isDragging) {
        const dx = e.clientX - this.lastMouseX;
        const dy = e.clientY - this.lastMouseY;
        this.viewport.x += dx;
        this.viewport.y += dy;
        this.lastMouseX = e.clientX;
        this.lastMouseY = e.clientY;
        this.updateTransform();
      }
    });
    window.addEventListener("mouseup", () => {
      this.isDragging = false;
      this.svgContainer.style.cursor = "grab";
    });
    this.wrapper.addEventListener("wheel", (e) => {
      e.preventDefault();
      const rect = this.wrapper.getBoundingClientRect();
      const mouseX = e.clientX - rect.left;
      const mouseY = e.clientY - rect.top;
      const factor = e.deltaY > 0 ? 0.9 : 1.1;
      this.zoomAt(mouseX, mouseY, factor);
    }, { passive: false });
    let lastTouchDistance = 0;
    let lastTouchX = 0;
    let lastTouchY = 0;
    this.wrapper.addEventListener("touchstart", (e) => {
      if (e.touches.length === 1) {
        lastTouchX = e.touches[0].clientX;
        lastTouchY = e.touches[0].clientY;
      } else if (e.touches.length === 2) {
        lastTouchDistance = Math.hypot(
          e.touches[1].clientX - e.touches[0].clientX,
          e.touches[1].clientY - e.touches[0].clientY
        );
      }
    });
    this.wrapper.addEventListener("touchmove", (e) => {
      e.preventDefault();
      if (e.touches.length === 1) {
        const dx = e.touches[0].clientX - lastTouchX;
        const dy = e.touches[0].clientY - lastTouchY;
        this.viewport.x += dx;
        this.viewport.y += dy;
        lastTouchX = e.touches[0].clientX;
        lastTouchY = e.touches[0].clientY;
        this.updateTransform();
      } else if (e.touches.length === 2) {
        const distance = Math.hypot(
          e.touches[1].clientX - e.touches[0].clientX,
          e.touches[1].clientY - e.touches[0].clientY
        );
        const factor = distance / lastTouchDistance;
        const centerX = (e.touches[0].clientX + e.touches[1].clientX) / 2;
        const centerY = (e.touches[0].clientY + e.touches[1].clientY) / 2;
        const rect = this.wrapper.getBoundingClientRect();
        this.zoomAt(centerX - rect.left, centerY - rect.top, factor);
        lastTouchDistance = distance;
      }
    }, { passive: false });
    this.svgContainer.addEventListener("click", (e) => {
      const target = e.target;
      if (target.id) {
        this.emit("click", { elementId: target.id, element: target });
      }
    });
  }
  zoomAt(x, y, factor) {
    const oldScale = this.viewport.scale;
    const newScale = Math.max(
      this.options.minZoom,
      Math.min(this.options.maxZoom, oldScale * factor)
    );
    this.viewport.x = x - (x - this.viewport.x) * (newScale / oldScale);
    this.viewport.y = y - (y - this.viewport.y) * (newScale / oldScale);
    this.viewport.scale = newScale;
    this.updateTransform();
  }
  updateTransform() {
    this.svgContainer.style.transform = `translate(${this.viewport.x}px, ${this.viewport.y}px) scale(${this.viewport.scale})`;
  }
  injectStyles() {
    if (document.getElementById("svj-viewer-styles")) return;
    const style = document.createElement("style");
    style.id = "svj-viewer-styles";
    style.textContent = `
      .svj-sensor-group {
        position: absolute;
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 2px;
        transform: translate(-50%, -50%);
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        font-size: 9px;
        pointer-events: auto;
      }
      .svj-sensor-room-name {
        background: rgba(0, 0, 0, 0.7);
        color: #fff;
        padding: 2px 4px;
        border-radius: 2px;
        font-weight: 600;
        font-size: 10px;
        white-space: nowrap;
      }
      .svj-sensor-values {
        display: flex;
        flex-wrap: wrap;
        gap: 2px;
        justify-content: center;
        max-width: 120px;
      }
      .svj-sensor {
        background: rgba(30, 30, 60, 0.9);
        color: #fff;
        padding: 2px 4px;
        border-radius: 2px;
        display: flex;
        align-items: center;
        gap: 2px;
        white-space: nowrap;
        border: 1px solid rgba(100, 100, 180, 0.5);
      }
      .svj-sensor-icon {
        font-size: 10px;
      }
      .svj-sensor-value {
        font-weight: 600;
      }
      .svj-sensor-unit {
        opacity: 0.7;
        font-size: 8px;
      }
      .svj-sensor--temperature .svj-sensor-value { color: #ff9800; }
      .svj-sensor--humidity .svj-sensor-value { color: #2196f3; }
      .svj-sensor--motion .svj-sensor-value { color: #4caf50; }
      .svj-sensor--light .svj-sensor-value { color: #ffeb3b; }
      .svj-sensor--power .svj-sensor-value { color: #e91e63; }
      .svj-sensor--unknown .svj-sensor-value { color: #9e9e9e; }
      .svj-sensor--loading {
        opacity: 0.5;
      }
      .svj-sensor--error {
        border-color: rgba(244, 67, 54, 0.7);
        background: rgba(244, 67, 54, 0.2);
      }
      .svj-sensor--error .svj-sensor-value {
        color: #f44336 !important;
      }
    `;
    document.head.appendChild(style);
  }
  getSensorIcon(sensorType) {
    const icons = {
      temperature: "\u{1F321}\uFE0F",
      humidity: "\u{1F4A7}",
      motion: "\u{1F6B6}",
      light: "\u{1F4A1}",
      power: "\u26A1",
      door: "\u{1F6AA}",
      window: "\u{1FA9F}",
      co2: "\u{1FAC1}",
      pressure: "\u{1F388}",
      battery: "\u{1F50B}"
    };
    return icons[sensorType.toLowerCase()] || "\u{1F4CA}";
  }
  /**
   * Load an SVJ project from a URL
   */
  async load(url) {
    try {
      this.project = await loadSvjFromUrl(url);
      this.currentFloorIndex = this.project.selectedFloorIndex;
      this.renderCurrentFloor();
      this.emit("load", { project: this.project });
    } catch (error) {
      this.emit("error", { error });
      throw error;
    }
  }
  /**
   * Load an SVJ project from a File object
   */
  async loadFile(file) {
    try {
      this.project = await loadSvjFromFile(file);
      this.currentFloorIndex = this.project.selectedFloorIndex;
      this.renderCurrentFloor();
      this.emit("load", { project: this.project });
    } catch (error) {
      this.emit("error", { error });
      throw error;
    }
  }
  /**
   * Load from an already-parsed project
   */
  loadProject(project) {
    this.project = project;
    this.currentFloorIndex = project.selectedFloorIndex;
    this.renderCurrentFloor();
    this.emit("load", { project });
  }
  renderCurrentFloor() {
    if (!this.project || this.project.floors.length === 0) {
      this.svgContainer.innerHTML = '<p style="padding: 20px; color: #666;">No floors to display</p>';
      return;
    }
    const floor = this.project.floors[this.currentFloorIndex];
    if (!floor) return;
    let svgContent = floor.svg;
    if (svgContent.startsWith("<?xml")) {
      svgContent = svgContent.replace(/<\?xml[^?]*\?>\s*/i, "");
    }
    this.svgContainer.innerHTML = svgContent;
    this.svgContainer.appendChild(this.sensorOverlay);
    const svg = this.svgContainer.querySelector("svg");
    if (svg) {
      svg.style.display = "block";
      const width = svg.getAttribute("width");
      const height = svg.getAttribute("height");
      const viewBox = svg.getAttribute("viewBox");
      if (!viewBox && width && height) {
        const w = parseFloat(width);
        const h = parseFloat(height);
        if (!isNaN(w) && !isNaN(h)) {
          svg.setAttribute("viewBox", `0 0 ${w} ${h}`);
        }
      }
      const vb = svg.getAttribute("viewBox");
      if (vb) {
        const parts = vb.split(/\s+|,/).map(Number);
        if (parts.length === 4) {
          svg.style.width = parts[2] + "px";
          svg.style.height = parts[3] + "px";
        }
      } else if (width && height) {
        svg.style.width = width.includes("px") ? width : width + "px";
        svg.style.height = height.includes("px") ? height : height + "px";
      }
      svg.removeAttribute("width");
      svg.removeAttribute("height");
    }
    this.renderSensors();
    if (this.options.fitOnLoad) {
      requestAnimationFrame(() => {
        this.fitToView();
      });
    }
  }
  renderSensors() {
    this.sensorOverlay.innerHTML = "";
    this.sensors.clear();
    if (!this.project) return;
    const floor = this.project.floors[this.currentFloorIndex];
    if (!floor?.config?.rooms) return;
    for (const [roomId, room] of Object.entries(floor.config.rooms)) {
      if (!room.sensors || Object.keys(room.sensors).length === 0) continue;
      const centerX = room.center_x ?? 0;
      const centerY = room.center_y ?? 0;
      const group = document.createElement("div");
      group.className = "svj-sensor-group";
      group.style.left = `${centerX}px`;
      group.style.top = `${centerY}px`;
      group.dataset.roomId = roomId;
      const roomLabel = document.createElement("div");
      roomLabel.className = "svj-sensor-room-name";
      roomLabel.textContent = room.name;
      group.appendChild(roomLabel);
      const valuesContainer = document.createElement("div");
      valuesContainer.className = "svj-sensor-values";
      for (const [sensorType, sensorConfig] of Object.entries(room.sensors)) {
        const sensorKey = `${roomId}:${sensorType}`;
        const sensorEl = this.createSensorElement(sensorType, sensorConfig);
        valuesContainer.appendChild(sensorEl);
        this.sensors.set(sensorKey, {
          roomId,
          sensorType,
          config: sensorConfig,
          element: sensorEl
        });
      }
      group.appendChild(valuesContainer);
      this.sensorOverlay.appendChild(group);
    }
  }
  createSensorElement(sensorType, config) {
    const el = document.createElement("div");
    const typeClass = this.getSensorTypeClass(sensorType);
    el.className = `svj-sensor svj-sensor--${typeClass} svj-sensor--loading`;
    el.dataset.sensorType = sensorType;
    const icon = document.createElement("span");
    icon.className = "svj-sensor-icon";
    icon.textContent = this.getSensorIcon(sensorType);
    const value = document.createElement("span");
    value.className = "svj-sensor-value";
    value.textContent = "--";
    const unit = document.createElement("span");
    unit.className = "svj-sensor-unit";
    unit.textContent = config.unit || this.getDefaultUnit(sensorType);
    el.appendChild(icon);
    el.appendChild(value);
    el.appendChild(unit);
    return el;
  }
  getSensorTypeClass(sensorType) {
    const known = ["temperature", "humidity", "motion", "light", "power", "door", "window", "co2", "pressure", "battery"];
    const lower = sensorType.toLowerCase();
    return known.includes(lower) ? lower : "unknown";
  }
  getDefaultUnit(sensorType) {
    const units = {
      temperature: "\xB0C",
      humidity: "%",
      motion: "",
      light: "lx",
      power: "W",
      co2: "ppm",
      pressure: "hPa",
      battery: "%"
    };
    return units[sensorType.toLowerCase()] || "";
  }
  /**
   * Fit the content to the viewport
   */
  fitToView(padding = 20) {
    const svg = this.svgContainer.querySelector("svg");
    if (!svg) return;
    const containerRect = this.wrapper.getBoundingClientRect();
    if (containerRect.width === 0 || containerRect.height === 0) return;
    let svgWidth = 0;
    let svgHeight = 0;
    const viewBox = svg.getAttribute("viewBox");
    if (viewBox) {
      const parts = viewBox.split(/\s+|,/).map(Number);
      if (parts.length === 4) {
        svgWidth = parts[2];
        svgHeight = parts[3];
      }
    }
    if (svgWidth === 0 || svgHeight === 0) {
      svgWidth = parseFloat(svg.style.width) || svg.clientWidth || 100;
      svgHeight = parseFloat(svg.style.height) || svg.clientHeight || 100;
    }
    if (svgWidth === 0 || svgHeight === 0) return;
    const availableWidth = containerRect.width - padding * 2;
    const availableHeight = containerRect.height - padding * 2;
    const scaleX = availableWidth / svgWidth;
    const scaleY = availableHeight / svgHeight;
    const scale = Math.min(scaleX, scaleY, 1);
    const scaledWidth = svgWidth * scale;
    const scaledHeight = svgHeight * scale;
    this.viewport.scale = scale;
    this.viewport.x = (containerRect.width - scaledWidth) / 2;
    this.viewport.y = (containerRect.height - scaledHeight) / 2;
    this.updateTransform();
  }
  /**
   * Reset zoom and position
   */
  resetView() {
    this.viewport = { x: 0, y: 0, scale: 1 };
    this.updateTransform();
  }
  /**
   * Set the current floor by index
   */
  setFloorIndex(index) {
    if (!this.project || index < 0 || index >= this.project.floors.length) {
      return;
    }
    const previousIndex = this.currentFloorIndex;
    this.currentFloorIndex = index;
    this.renderCurrentFloor();
    this.emit("floorChange", {
      floor: this.project.floors[index],
      index,
      previousIndex
    });
  }
  /**
   * Set the current floor by ID
   */
  setFloor(id) {
    if (!this.project) return;
    const index = this.project.floors.findIndex((f) => f.id === id);
    if (index !== -1) {
      this.setFloorIndex(index);
    }
  }
  /**
   * Get the current floor
   */
  getCurrentFloor() {
    return this.project?.floors[this.currentFloorIndex] ?? null;
  }
  /**
   * Get all floors
   */
  getFloors() {
    return this.project?.floors ?? [];
  }
  /**
   * Get the project
   */
  getProject() {
    return this.project;
  }
  /**
   * Get current zoom level
   */
  getZoom() {
    return this.viewport.scale;
  }
  /**
   * Set zoom level
   */
  setZoom(scale) {
    const rect = this.wrapper.getBoundingClientRect();
    const centerX = rect.width / 2;
    const centerY = rect.height / 2;
    const factor = scale / this.viewport.scale;
    this.zoomAt(centerX, centerY, factor);
  }
  /**
   * Add event listener
   */
  on(event, handler) {
    if (!this.eventHandlers.has(event)) {
      this.eventHandlers.set(event, /* @__PURE__ */ new Set());
    }
    this.eventHandlers.get(event).add(handler);
  }
  /**
   * Remove event listener
   */
  off(event, handler) {
    this.eventHandlers.get(event)?.delete(handler);
  }
  emit(event, data) {
    this.eventHandlers.get(event)?.forEach((handler) => handler(data));
  }
  /**
   * Update a sensor value
   * @param roomId The room ID
   * @param sensorType The sensor type (e.g., 'temperature', 'humidity')
   * @param value The sensor value
   * @param unit Optional unit override
   */
  setSensorValue(roomId, sensorType, value, unit) {
    const sensorKey = `${roomId}:${sensorType}`;
    const sensor = this.sensors.get(sensorKey);
    if (!sensor) return;
    sensor.value = {
      value,
      unit: unit ?? sensor.config.unit ?? this.getDefaultUnit(sensorType),
      timestamp: /* @__PURE__ */ new Date()
    };
    const valueEl = sensor.element.querySelector(".svj-sensor-value");
    const unitEl = sensor.element.querySelector(".svj-sensor-unit");
    if (valueEl) {
      valueEl.textContent = String(value);
    }
    if (unitEl && sensor.value.unit) {
      unitEl.textContent = sensor.value.unit;
    }
    sensor.element.classList.remove("svj-sensor--loading");
    this.emit("sensorUpdate", {
      roomId,
      sensorType,
      value: sensor.value
    });
  }
  /**
   * Update multiple sensor values at once
   * @param values Map of "roomId:sensorType" to value
   */
  setSensorValues(values) {
    for (const [key, value] of Object.entries(values)) {
      const [roomId, sensorType] = key.split(":");
      if (roomId && sensorType) {
        this.setSensorValue(roomId, sensorType, value);
      }
    }
  }
  /**
   * Get all sensor configurations for the current floor
   */
  getSensors() {
    return Array.from(this.sensors.values()).map((s) => ({
      roomId: s.roomId,
      sensorType: s.sensorType,
      config: s.config,
      value: s.value
    }));
  }
  /**
   * Get sensor value
   */
  getSensorValue(roomId, sensorType) {
    const sensorKey = `${roomId}:${sensorType}`;
    return this.sensors.get(sensorKey)?.value;
  }
  /**
   * Start polling sensor values at the configured interval
   */
  startPolling() {
    if (this.isPolling) return;
    if (!this.project) return;
    const interval = (this.project.config.poll_interval ?? 30) * 1e3;
    this.isPolling = true;
    this.fetchAllSensors();
    this.pollingInterval = setInterval(() => {
      this.fetchAllSensors();
    }, interval);
  }
  /**
   * Stop polling sensor values
   */
  stopPolling() {
    if (this.pollingInterval) {
      clearInterval(this.pollingInterval);
      this.pollingInterval = null;
    }
    this.isPolling = false;
  }
  /**
   * Fetch all sensor values from their configured URLs
   */
  async fetchAllSensors() {
    const fetchPromises = [];
    for (const [sensorKey, sensor] of this.sensors) {
      if (sensor.config.url) {
        fetchPromises.push(this.fetchSensorValue(sensorKey, sensor));
      }
    }
    await Promise.allSettled(fetchPromises);
  }
  /**
   * Fetch a single sensor value
   */
  async fetchSensorValue(sensorKey, sensor) {
    if (!sensor.config.url) return;
    try {
      const response = await fetch(sensor.config.url, {
        method: "GET",
        headers: {
          "Accept": "application/json"
        }
      });
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
      }
      const data = await response.json();
      let value;
      if (typeof data === "number" || typeof data === "string") {
        value = data;
      } else if (data && typeof data === "object") {
        value = data.value ?? data.state ?? data.reading ?? data.result ?? "--";
      } else {
        value = "--";
      }
      sensor.element.classList.remove("svj-sensor--error", "svj-sensor--loading");
      this.setSensorValue(sensor.roomId, sensor.sensorType, value);
    } catch (error) {
      sensor.element.classList.remove("svj-sensor--loading");
      sensor.element.classList.add("svj-sensor--error");
      const valueEl = sensor.element.querySelector(".svj-sensor-value");
      if (valueEl) {
        valueEl.textContent = "ERR";
      }
      this.emit("error", {
        type: "sensor",
        roomId: sensor.roomId,
        sensorType: sensor.sensorType,
        error
      });
    }
  }
  /**
   * Check if polling is active
   */
  isPollingActive() {
    return this.isPolling;
  }
  /**
   * Destroy the viewer and clean up
   */
  destroy() {
    this.stopPolling();
    this.wrapper.remove();
    this.eventHandlers.clear();
    this.sensors.clear();
  }
};
// Annotate the CommonJS export names for ESM import in node:
0 && (module.exports = {
  SvjViewer,
  entryAsString,
  getEntriesMatching,
  getEntryByName,
  loadSvjFromFile,
  loadSvjFromUrl,
  parseSvj,
  parseTar
});
